# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

Sufee is a responsive Bootstrap 5 Admin Dashboard Template built with HTML, CSS/SCSS, and JavaScript. Originally a Bootstrap 4 template, it has been completely modernized in v2.0 with Bootstrap 5, Vite build system, and a component-based architecture. It's a static front-end template with no backend components, designed for creating admin interfaces and dashboards.

### Version 2.2 Changes

- Updated all dependencies to their latest versions
- Bootstrap 5.3.8 (latest stable release)
- Font Awesome 7.1.0 (major upgrade with new icons)
- Vite 7.1.8 (enhanced build performance)
- ESLint 9.36.0, Sass 1.93.2, and other dev tools updated
- Zero security vulnerabilities across all dependencies

### Version 2.1 Changes

- ESLint & Prettier integration for code quality and formatting
- Professional error handling with custom 404/500 pages
- Self-hosted fonts for privacy and GDPR compliance
- Global error handler with user-friendly notifications
- Enhanced build process with error page integration

### Version 2.0 Changes

- Migrated from Bootstrap 4 to Bootstrap 5
- Replaced Grunt/Bower with Vite/NPM
- Added dynamic partials system for code reusability
- Converted to ES6 modules
- Removed Internet Explorer support
- Fixed sidebar animations and navigation issues
- Improved performance with modern build tools

## Build System & Commands

This project uses Vite for build automation and development:

### Development Commands

- `npm run dev` - Starts the Vite development server with hot module replacement on port 3001
- `npm run build` - Builds the project for production
- `npm run preview` - Previews the production build locally
- `npm run lint` - Lint and auto-fix JavaScript code with ESLint
- `npm run lint:check` - Check for linting errors without fixing
- `npm run format` - Format all files with Prettier
- `npm run format:check` - Check formatting without modifying files
- `npm run quality:fix` - Run both linting and formatting (recommended before commits)
- `npm run quality` - Check both linting and formatting without changes

### Build Pipeline

1. **Development**: Vite dev server with HMR for instant updates
2. **SCSS Compilation**: Automatic SCSS compilation with modern-compiler API
3. **Module Resolution**: ES modules with optimized dependency pre-bundling
4. **Production Build**: Minification, tree-shaking, and code splitting
5. **Legacy Support**: Optional legacy browser support via @vitejs/plugin-legacy

### Code Quality Tools

The project includes modern code quality tools:

- **ESLint**: JavaScript linting with modern ES6+ rules and best practices
- **Prettier**: Automatic code formatting with consistent style
- **Configuration**: Custom rules defined in `eslint.config.js` and `.prettierrc.json`
- **Pre-commit**: Run `npm run quality:fix` before committing to ensure code quality

### Dependency Management

All dependencies are now managed through NPM. The project has migrated from Bower/Grunt to Vite.
Key dependencies include Bootstrap 5, Chart.js, and various visualization libraries. The project is 100% jQuery-free, using modern vanilla JavaScript throughout.

## Architecture & Structure

### Core Structure

- **Static HTML Templates**: Multiple HTML pages in src/ directory (index.html, ui-*.html, etc.)
- **SCSS Architecture**: Modular SCSS with partials in src/styles/
- **JavaScript Components**: Main entry point in src/main.js with modular components
- **Vendor Dependencies**: All managed through NPM and imported via Vite
- **Multi-page Application**: Each HTML file is a separate entry point configured in `vite.config.js`

### Application Architecture

The application follows a component-based architecture:

1. **App Class** (`src/scripts/app.js`): Main application controller
   - Manages component lifecycle
   - Handles sidebar state and responsive behavior
   - Provides global utilities (notifications, keyboard shortcuts)
   - Dynamically loads page-specific components

2. **Partials Loader** (`src/scripts/partials-loader.js`): Dynamic HTML component system
   - Loads reusable HTML partials (header, sidebar, etc.)
   - Manages active navigation states
   - Generates dynamic breadcrumbs

3. **Component Modules** (`src/scripts/components/`): Self-contained UI components
   - Charts, DataTables, forms, widgets, etc.
   - Each component handles its own initialization and cleanup
   - Lazy-loaded based on page requirements

4. **Utilities** (`src/scripts/utils/`): Shared helper functions
   - DOM utilities, error handling, etc.

### Key Directories

- `src/` - Source files including HTML templates, styles, and scripts
- `src/styles/` - SCSS source files with modular architecture
- `src/scripts/` - JavaScript modules and components
- `public/` - Static assets served directly (favicons, fonts, etc.)
- `dist/` - Production build output (generated by Vite)
- `images/` - Static images and assets

### SCSS Architecture

The main stylesheet (`src/styles/main.scss`) imports:

- `_variables.scss` - Theme variables and configuration
- `_gauge.scss` - Gauge chart styles
- `_switches.scss` - Toggle switch components
- `_widgets.scss` - Dashboard widget styles
- `_socials.scss` - Social media components

### Layout System

- **Left Panel**: Collapsible sidebar navigation (`aside.left-panel`)
- **Right Panel**: Main content area (`.right-panel`)
- **Header**: Top navigation with search and user menu
- **Responsive**: Mobile-first design with breakpoint-specific styles

### Key Libraries & Dependencies

- Bootstrap 5 (UI framework)
- NO jQuery - 100% vanilla JavaScript
- Chart.js v4 (modern charting library)
- DataTables with Bootstrap 5 integration
- Font Awesome 6 (icons)
- Themify Icons (additional icons)
- Flag Icons (country flags)
- Various visualization libraries (Flot, Peity, JQVMap, GaugeJS)
- Form validation (native HTML5 validation)
- Animation library (Animate.css)
- Select enhancement (Chosen.js)
- Maps integration (GMaps.js)

## Development Notes

### Working with Styles

- Always edit SCSS files in `src/styles/`, never directly edit compiled CSS
- Vite automatically compiles SCSS on save with HMR
- Source maps are included for debugging in development

### Adding New Components

- Follow existing patterns in the SCSS partial structure
- Use Bootstrap 5 classes and utilities
- Maintain the existing color scheme defined in `_variables.scss`
- Import new dependencies through NPM and configure in Vite

### JavaScript Integration

- Main entry point is `src/main.js` which imports all dependencies and initializes the application
- Component scripts are organized in `src/scripts/components/`
- Utility functions are in `src/scripts/utils/`
- Core application logic is in `src/scripts/app.js` (App class)
- All vendor libraries are imported as ES modules
- All JavaScript is modern ES6+ modules with no jQuery dependency
- Bootstrap, Chart.js, and error handler are exposed globally in development mode for debugging

### Browser Support

- CSS is post-processed for compatibility with last 4 versions of major browsers
- IE support has been removed as Internet Explorer was discontinued in 2022
- The `<meta http-equiv="X-UA-Compatible" content="IE=edge">` tag should not be used in any HTML files

### Vite Configuration

The Vite build is configured with:

- **Root directory**: `src/` (all source files)
- **Multi-page setup**: Each HTML page is a separate entry point in `rollupOptions.input`
- **Path aliases**: `@`, `@styles`, `@scripts`, `@assets` for clean imports
- **SCSS**: Modern compiler API for fast compilation
- **Static copy**: Partials directory copied to build output
- **Dev server**: Runs on port 3001 with auto-open enabled
- **Production**: Source maps, code splitting, and minification enabled

When adding new HTML pages, add them to the `rollupOptions.input` object in `vite.config.js`.

This is a front-end only template - no server-side components or API integrations are included.

## Partials System (New)

The template now includes a modern partials system for reusable components, eliminating code duplication across HTML files.

### Key Features

- **Dynamic HTML Loading**: Common components are loaded at runtime
- **Cache System**: Partials are cached after first load for performance
- **Active State Management**: Automatic sidebar navigation highlighting
- **Breadcrumb System**: Dynamic breadcrumb generation
- **Bootstrap 5 Integration**: Full Bootstrap 5 compatibility

### Core Partials

#### Available Partials

- `head-common.html` - Common meta tags, fonts, and favicon
- `sidebar.html` - Left navigation sidebar with menu items
- `header.html` - Top navigation with search, notifications, and user menu
- `scripts-common.html` - Common JavaScript imports

#### Using Partials in HTML Files

Add these data attributes to div elements where you want partials to load:

```html
<!-- Load common head elements -->
<div data-partial="head-common"></div>

<!-- Load and replace with sidebar -->
<div data-partial="sidebar" data-partial-replace="true"></div>

<!-- Load and replace with header -->
<div data-partial="header" data-partial-replace="true"></div>

<!-- Dynamic breadcrumbs -->
<div data-breadcrumb data-breadcrumb-title="Page Title" data-breadcrumb-path="Parent|Child|Current"></div>
```

#### Data Attributes

- `data-partial="name"` - Specifies which partial to load
- `data-partial-replace="true"` - Replaces the div entirely (default: false, appends content)
- `data-breadcrumb` - Generates dynamic breadcrumbs
- `data-breadcrumb-title` - Page title for breadcrumbs
- `data-breadcrumb-path` - Breadcrumb path separated by |
- `data-page="page-name"` - Used for sidebar active state highlighting

### Partials Architecture

#### Directory Structure

```text
src/
├── partials/
│   ├── head-common.html      # Common head elements
│   ├── sidebar.html          # Navigation sidebar
│   ├── header.html           # Top header
│   └── scripts-common.html   # Common scripts
├── scripts/
│   ├── app.js                # Main application class
│   ├── partials-loader.js    # Main partials loading system
│   ├── breadcrumb-helper.js  # Breadcrumb generation
│   ├── components/           # UI component modules
│   │   ├── charts.js         # Chart.js integration
│   │   ├── datatable.js      # DataTables integration
│   │   ├── navigation.js     # Navigation component
│   │   ├── theme-manager.js  # Theme switching
│   │   ├── user-menu.js      # User menu component
│   │   ├── validation.js     # Form validation
│   │   ├── widgets.js        # Dashboard widgets
│   │   └── ...
│   └── utils/                # Utility functions
│       ├── dom.js            # DOM utilities
│       └── error-handler.js  # Global error handling
└── main.js                   # Entry point with partials initialization
```

#### Partials Loader System

The `partials-loader.js` module provides:

- **PartialsLoader Class**: Main loader with caching
- **loadPartial(name)**: Loads individual partials with cache
- **loadAllPartials()**: Processes all `data-partial` elements
- **initializeSidebarActiveState()**: Highlights active menu items
- **initializeHeader()**: Sets up header functionality

#### Integration in main.js

The application initializes in this order:

```javascript
import { App } from './scripts/app.js'
import { partialsLoader } from './scripts/partials-loader.js'

// 1. Create and initialize App instance
const app = new App()
app.init()

// 2. Load all partials (header, sidebar, etc.)
partialsLoader.loadAllPartials()
partialsLoader.initializeSidebarActiveState()
partialsLoader.initializeHeader()
```

The App class handles:

- Component lifecycle management
- Sidebar toggle functionality (desktop collapse, mobile slide)
- Responsive behavior and state persistence
- Keyboard shortcuts (Ctrl+K for search, Ctrl+\ for sidebar toggle, Esc to close)
- Dynamic component loading based on page type
- Global notification system

### Creating New Partials

1. **Create the partial file** in `src/partials/filename.html`
2. **Add to HTML** using `<div data-partial="filename"></div>`
3. **No code changes needed** - the system auto-discovers new partials

### Sidebar Active States

The system automatically highlights active menu items based on the `data-page` attribute on the body element:

```html
<body class="sufee-dashboard" data-page="ui-buttons">
```

Sidebar menu items with matching `data-page` attributes receive the `active` class.

### Benefits

- **No Code Duplication**: Write header/sidebar once, use everywhere
- **Easy Maintenance**: Update one file to change all pages
- **Performance**: Cached loading after first request
- **Bootstrap 5 Compatible**: Modern Bootstrap 5 components
- **SEO Friendly**: Proper semantic HTML structure
- **IE Support Removed**: Modernized codebase without legacy IE compatibility

### Migration from Old System

All HTML files in `src/` have been updated to use the partials system. The old inline header/sidebar code has been replaced with data attributes for dynamic loading.
